#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include "hxd.h"
#include "xmalloc.h"

/* in files.c */
extern int hldir_to_path (struct hl_data_hdr *dh, const char *root, char *rlpath, char *pathp);

/* categories are directories with names starting with "cat_" ... */
static int
cat_to_path (struct hl_data_hdr *dh, const char *newsdir, char *rlpath, char *path)
{
	char catpath[MAXPATHLEN];

	/* XXX dirty */
	hldir_to_path(dh, newsdir, rlpath, path);
	snprintf(catpath, sizeof(catpath), "%s/cat_%s",
		 newsdir, basename(path));
	if (!realpath(catpath, rlpath))
		return errno;

	return 0;
}

static void
snd_news_dirlist (struct htlc_conn *htlc, struct hl_newslist_hdr **fhdrs, u_int16_t fhdrcount)
{
	struct hl_hdr h;
	u_int16_t i;
	u_int32_t pos, this_off, len = SIZEOF_HL_HDR;

	h.type = htonl(HTLS_HDR_TASK);
	h.trans = htonl(htlc->trans);
	htlc->trans++;
	h.flag = 0;
	h.hc = htons(fhdrcount);

	for (i = 0; i < fhdrcount; i++)
		len += SIZEOF_HL_DATA_HDR + ntohs(fhdrs[i]->len);
	h.len = htonl(len - (SIZEOF_HL_HDR - sizeof(h.hc)));
	h.len2 = h.len;
	pos = htlc->out.pos + htlc->out.len;
	this_off = pos;
	qbuf_set(&htlc->out, htlc->out.pos, htlc->out.len + len);
	memcpy(&htlc->out.buf[pos], &h, SIZEOF_HL_HDR);
	pos += SIZEOF_HL_HDR;
	FD_SET(htlc->fd, &hxd_wfds);
	for (i = 0; i < fhdrcount; i++) {
		memcpy(&htlc->out.buf[pos], fhdrs[i], SIZEOF_HL_DATA_HDR + ntohs(fhdrs[i]->len));
		pos += SIZEOF_HL_DATA_HDR + ntohs(fhdrs[i]->len);
	}
	len = pos - this_off;
#ifdef CONFIG_COMPRESS
	if (htlc->compress_encode_type != COMPRESS_NONE)
		len = compress_encode(htlc, this_off, len);
#endif
#ifdef CONFIG_CIPHER
	if (htlc->cipher_encode_type != CIPHER_NONE)
		cipher_encode(htlc, this_off, len);
#endif
}

/* ref: hxd_scandir */
void
tnews_send_dirlist (struct htlc_conn *htlc, const char *path)
{
	u_int8_t nlen, ntype;
	u_int16_t count = 0, maxcount = 0;
	struct hl_newslist_hdr **fhdrs = 0;
	DIR *dir;
	struct dirent *de;
	struct stat sb;
	char pathbuf[MAXPATHLEN];

	if (!(dir = opendir(path))) {
		snd_strerror(htlc, errno);
		return;
	}
	while ((de = readdir(dir))) {
		/* skip files whose names start with '.' */
		if (de->d_name[0] == '.')
			continue;
		snprintf(pathbuf, sizeof(pathbuf), "%s/%s", path, de->d_name);
		if (SYS_lstat(pathbuf, &sb)) {
			continue;
		}
		/* skip files */
		if (!S_ISDIR(sb.st_mode))
			continue;
		if (count >= maxcount) {
			maxcount += 16;
			fhdrs = xrealloc(fhdrs, sizeof(struct hl_newslist_hdr *) * maxcount);
		}
		nlen = (u_int8_t)strlen(de->d_name);
		if (!strncmp(de->d_name, "cat_", 4)) {
			nlen -= 4;
			ntype = 10;
		} else {
			ntype = 1;
		}
		fhdrs[count] = xmalloc(SIZEOF_HL_NEWSLIST_HDR + nlen);
		fhdrs[count]->type = htons(HTLS_DATA_NEWS_DIRLIST);
		fhdrs[count]->len = htons((u_int16_t)nlen+1);
		fhdrs[count]->ntype = ntype;
		if (ntype == 1)
			memcpy(fhdrs[count]->name, de->d_name, nlen);
		else
			memcpy(fhdrs[count]->name, de->d_name+4, nlen);
		count++;
	}
	closedir(dir);
	snd_news_dirlist(htlc, fhdrs, count);
	while (count) {
		count--;
		xfree(fhdrs[count]);
	}
	xfree(fhdrs);
}

static void
snd_news_catlist (struct htlc_conn *htlc, struct hl_news_thread_hdr **fhdrs, u_int16_t fhdrcount)
{
	struct hl_news_threadlist_hdr *tlh;
	struct hl_news_thread_hdr *fh;
	struct hl_hdr h;
	u_int16_t i;
	u_int32_t pos, this_off, len = SIZEOF_HL_HDR;

	h.type = htonl(HTLS_HDR_TASK);
	h.trans = htonl(htlc->trans);
	htlc->trans++;
	h.flag = 0;
	h.hc = htons(fhdrcount);

	len += SIZEOF_HL_NEWS_THREADLIST_HDR;
	for (i = 0; i < fhdrcount; i++)
		len += SIZEOF_HL_NEWS_THREAD_HDR + 5 + fhdrs[i]->__x0;
	h.len = htonl(len - (SIZEOF_HL_HDR - sizeof(h.hc)));
	h.len2 = h.len;
	pos = htlc->out.pos + htlc->out.len;
	this_off = pos;
	qbuf_set(&htlc->out, htlc->out.pos, htlc->out.len + len);
	memcpy(&htlc->out.buf[pos], &h, SIZEOF_HL_HDR);
	pos += SIZEOF_HL_HDR;
	tlh = (struct hl_news_threadlist_hdr *)&htlc->out.buf[pos];
	tlh->type = htons(HTLS_DATA_NEWS_CATLIST);
	tlh->len = htons((u_int16_t)len - (SIZEOF_HL_HDR + SIZEOF_HL_DATA_HDR));
	tlh->__x0 = 0;
	tlh->thread_count = htonl(fhdrcount);
	tlh->__x1 = 0;
	pos += SIZEOF_HL_NEWS_THREADLIST_HDR;
	FD_SET(htlc->fd, &hxd_wfds);
	for (i = 0; i < fhdrcount; i++) {
		memcpy(&htlc->out.buf[pos], fhdrs[i], SIZEOF_HL_NEWS_THREAD_HDR + 5 + fhdrs[i]->__x0);
		fh = (struct hl_news_thread_hdr *)&htlc->out.buf[pos];
		fh->__x0 = 0; /* XXX */
		pos += SIZEOF_HL_NEWS_THREAD_HDR + 5 + fhdrs[i]->__x0;
	}
	len = pos - this_off;
#ifdef CONFIG_COMPRESS
	if (htlc->compress_encode_type != COMPRESS_NONE)
		len = compress_encode(htlc, this_off, len);
#endif
#ifdef CONFIG_CIPHER
	if (htlc->cipher_encode_type != CIPHER_NONE)
		cipher_encode(htlc, this_off, len);
#endif
}

struct mac_date {
	u_int16_t base_year PACKED;
	u_int16_t pad PACKED;
	u_int32_t seconds PACKED;
};

static int
read_newsfile (const char *pathbuf, u_int8_t *poster, u_int8_t *mimetype, u_int8_t *subject,
	       u_int8_t *date, u_int32_t *id, u_int32_t *parent_id, u_int16_t *datasize, u_int8_t **datap)
{
	int fd, r, lastlinelen;
	char buf[1024], *p, *lastlinep, *nextp;
	size_t len, totalread = 0;
	struct stat sb;
	u_int16_t ds;

	fd = SYS_open(pathbuf, O_RDONLY, 0);
	if (fd < 0)
		return errno;
	if (fstat(fd, &sb)) {
		close(fd);
		return errno;
	}
	*id = *parent_id = 0;
	*datasize = 0;
	poster[0] = mimetype[0] = subject[0];
	memset(date, 0, 8);
	p = lastlinep = buf;
	for (;;) {
		lastlinelen = p - lastlinep;
		if (lastlinelen) {
			memcpy(buf, lastlinep, lastlinelen);
			if (sizeof(buf)-lastlinelen-1 == 0)
				break;
		}
		r = read(fd, buf+lastlinelen, sizeof(buf)-lastlinelen-1);
		if (r <= 0)
			break;
		buf[r] = 0;
		for (p = buf; *p; p = nextp) {
			nextp = strchr(p, '\n');
			if (nextp) {
				*nextp = 0;
				nextp++;
				totalread += nextp - p;
				if (nextp == p+1) {
					/* end of header */
					if ((sb.st_size - totalread) > 0xffff)
						ds = 0xffff;
					else
						ds = sb.st_size - totalread;
					*datasize = ds;
					if (datap) { /* XXX */
						*datap = xmalloc(ds);
						lseek(fd, (off_t)totalread, SEEK_SET);
						read(fd, *datap, ds);
					}
					close(fd);
					return 0;
				}
				lastlinep = nextp;
			} else {
				do { totalread++; p++; } while (*p);
				break;
			}
			if (!strncasecmp(p, "FROM: ", 6)) {
				len = strlen(p+6);
				if (len > 255)
					len = 255;
				poster[0] = (u_int8_t)len;
				memcpy(poster+1, p+6, len);
			} else if (!strncasecmp(p, "CONTENT-TYPE: ", 14)) {
				len = strlen(p+14);
				if (len > 255)
					len = 255;
				mimetype[0] = (u_int8_t)len;
				memcpy(mimetype+1, p+14, len);
			} else if (!strncasecmp(p, "SUBJECT: ", 9)) {
				len = strlen(p+9);
				if (len > 255)
					len = 255;
				subject[0] = (u_int8_t)len;
				memcpy(subject+1, p+9, len);
			} else if (!strncasecmp(p, "DATE: ", 6)) {
				struct mac_date md;
				struct tm tm;
				time_t tt;

				memset(&tm, 0, sizeof(struct tm));
				/* XXX %z is GNU extension, may have locale troubles with %a */
				strptime(p+6, "%a, %d %b %Y %H:%M:%S %z", &tm);
				tt = mktime(&tm);
				/* md.base_year = htons((u_int16_t)(tm.tm_year + 1900)); */
				md.base_year = htons(1904);
				md.pad = 0;
				md.seconds = htonl((u_int32_t)(tt + 2082844800U));
				memcpy(date, &md, 8);
			} else if (!strncasecmp(p, "MESSAGE-ID: ", 12)) {
				*id = atou32(p+12);
			} else if (!strncasecmp(p, "REFERENCES: ", 12)) {
				*parent_id = atou32(p+12);
			}
		}
	}
	close(fd);

	return 0;
}

static void
tnews_send_catlist (struct htlc_conn *htlc, const char *path)
{
	u_int8_t *p, slen, plen, mlen, date[8], subject[256], poster[256], mimetype[256];
	u_int32_t id, parent_id;
	u_int16_t datasize;
	u_int16_t count = 0, maxcount = 0;
	struct hl_news_thread_hdr **fhdrs = 0, *th;
	DIR *dir;
	struct dirent *de;
	struct stat sb;
	char pathbuf[MAXPATHLEN];
	int err;

	if (!(dir = opendir(path))) {
		snd_strerror(htlc, errno);
		return;
	}
	while ((de = readdir(dir))) {
		/* skip files whose names start with '.' */
		if (de->d_name[0] == '.')
			continue;
		snprintf(pathbuf, sizeof(pathbuf), "%s/%s", path, de->d_name);
		if (SYS_lstat(pathbuf, &sb)) {
			continue;
		}
		/* skip directories */
		if (S_ISDIR(sb.st_mode))
			continue;
		if (count >= maxcount) {
			maxcount += 16;
			fhdrs = xrealloc(fhdrs, sizeof(struct hl_news_thread_hdr *) * maxcount);
		}
		err = read_newsfile(pathbuf, poster, mimetype, subject, date, &id, &parent_id, &datasize, 0);
		if (err)
			continue;
		slen = subject[0];
		plen = poster[0];
		mlen = mimetype[0];
		fhdrs[count] = xmalloc(SIZEOF_HL_NEWS_THREAD_HDR + 5 + slen + plen + mlen);
		th = fhdrs[count];
		th->id = htonl(id);
		memcpy(&th->date, date, 8);
		th->parent_id = htonl(parent_id);
		th->__x0 = 0;
		th->part_count = htons(1);
		/* XXX: use __x0 to store length */
		th->__x0 = slen + plen + mlen;
		p = th->data;
		memcpy(p, subject, slen+1);
		p += slen+1;
		memcpy(p, poster, plen+1);
		p += plen+1;
		memcpy(p, mimetype, mlen+1);
		p += mlen+1;
		S16HTON(datasize, p);
		count++;
	}
	closedir(dir);
	snd_news_catlist(htlc, fhdrs, count);
	while (count) {
		count--;
		xfree(fhdrs[count]);
	}
	xfree(fhdrs);
}

void
rcv_news_listdir (struct htlc_conn *htlc)
{
	char rlpath[MAXPATHLEN], path[MAXPATHLEN];
	int err;

	if (htlc->in.pos == SIZEOF_HL_HDR) {
		tnews_send_dirlist(htlc, hxd_cfg.paths.newsdir);
		return;
	}
	dh_start(htlc)
		if (dh_type != HTLC_DATA_NEWS_DIR)
			continue;
		if ((err = hldir_to_path(dh, hxd_cfg.paths.newsdir, rlpath, path))) {
			snd_strerror(htlc, err);
			return;
		}
		tnews_send_dirlist(htlc, rlpath);
	dh_end()
}

void
rcv_news_listcat (struct htlc_conn *htlc)
{
	char rlpath[MAXPATHLEN], path[MAXPATHLEN];
	int err;

	dh_start(htlc)
		if (dh_type != HTLC_DATA_NEWS_DIR)
			continue;
		if ((err = cat_to_path(dh, hxd_cfg.paths.newsdir, rlpath, path))) {
			snd_strerror(htlc, err);
			return;
		}
		tnews_send_catlist(htlc, rlpath);
	dh_end()
}

static void
tnews_send_thread (struct htlc_conn *htlc, const char *rlpath, u_int32_t tid)
{
	char pathbuf[MAXPATHLEN];
	int err;
	u_int8_t *data, date[8], subject[256], poster[256], mimetype[256];
	u_int32_t id, parent_id, lastid;
	u_int32_t ptid, ntid, nstid, partid32, ptid32, ntid32;
	u_int16_t datasize, slen, plen, mlen;
	DIR *dir;
	struct dirent *de;

	ptid = ntid = 0;
/* XXX get prev/next tids */
	if (!(dir = opendir(rlpath))) {
		goto xxx;
	}
	lastid = 0;
	while ((de = readdir(dir))) {
		if (de->d_name[0] == '.')
			continue;
		id = atou32(de->d_name);
		if (id == tid) {
			ptid = lastid;
		} else if (lastid == tid) {
			ntid = id;
			break;
		}
		lastid = id;
	}
	closedir(dir);
xxx:
	snprintf(pathbuf, sizeof(pathbuf), "%s/%u", rlpath, tid);
	err = read_newsfile(pathbuf, poster, mimetype, subject, date, &id, &parent_id, &datasize, &data);
	if (err) {
		snd_strerror(htlc, err);
		return;
	}
	if (data)
		LF2CR(data, datasize);
	slen = (u_int16_t)subject[0];
	plen = (u_int16_t)poster[0];
	mlen = (u_int16_t)mimetype[0];
	ptid32 = htonl(ptid);
	ntid32 = htonl(ntid);
	nstid = htonl(0);
	partid32 = htonl(parent_id);
	hlwrite(htlc, HTLS_HDR_TASK, 0, 9,
		HTLS_DATA_NEWS_POST, datasize, data,
		HTLS_DATA_NEWS_PREVTHREADID, sizeof(ptid32), &ptid32,
		HTLS_DATA_NEWS_NEXTTHREADID, sizeof(ntid32), &ntid32,
		HTLS_DATA_NEWS_PARENTTHREADID, sizeof(partid32), &partid32,
		HTLS_DATA_NEWS_NEXTSUBTHREADID, sizeof(nstid), &nstid,
		HTLS_DATA_NEWS_SUBJECT, slen, subject+1,
		HTLS_DATA_NEWS_POSTER, plen, poster+1,
		HTLS_DATA_NEWS_MIMETYPE, mlen, mimetype+1,
		HTLS_DATA_NEWS_DATE, 8, date);
	if (data)
		xfree(data);
}

void
rcv_news_get_thread (struct htlc_conn *htlc)
{
	char rlpath[MAXPATHLEN], path[MAXPATHLEN];
	int err;
	u_int16_t tid;

	tid = 0;
	dh_start(htlc)
		switch (dh_type) {
			case HTLC_DATA_NEWS_DIR:
				if ((err = cat_to_path(dh, hxd_cfg.paths.newsdir, rlpath, path))) {
					snd_strerror(htlc, err);
					return;
				}
				break;
			case HTLC_DATA_NEWS_THREADID:
				dh_getint(tid);
				break;
			case HTLC_DATA_NEWS_MIMETYPE:
				break;
		}
	dh_end()

	tnews_send_thread(htlc, rlpath, tid);
}

static void
get_gmtime (struct tm *tm)
{
	time_t t;

	t = time(0);
	/* XXX gmtime_r */
	localtime_r(&t, tm);
}

static int
write_newsfile (const char *rlpath, u_int32_t partid, u_int32_t replytid,
		u_int8_t *poster, u_int8_t *mimetype, u_int8_t *subject, u_int16_t postlen, u_int8_t *postp)
{
	char linebuf[1024], datestr[128];
	size_t linelen;
	char pathbuf[MAXPATHLEN];
	int fd, r;
	DIR *dir;
	struct dirent *de;
	u_int32_t tid, id, highid;
	struct tm tm;

	if (!(dir = opendir(rlpath)))
		return -1;
	highid = 0;
	while ((de = readdir(dir))) {
		if (de->d_name[0] == '.')
			continue;
		id = atou32(de->d_name);
		if (id > highid)
			highid = id;
	}
	closedir(dir);
	if (highid == 0xffffffff) {
		return -1;
	}
	tid = highid+1;
	snprintf(pathbuf, sizeof(pathbuf), "%s/%u", rlpath, tid);
	fd = SYS_open(pathbuf, O_WRONLY|O_CREAT, hxd_cfg.permissions.news_files);
	if (fd < 0)
		return -1;
	get_gmtime(&tm);
	strftime(datestr, sizeof(datestr), "%a, %d %b %Y %H:%M:%S %z", &tm);
	linelen = snprintf(linebuf, sizeof(linebuf),
			  "From: %s\n"
			  "Content-Type: %s\n"
			  "Subject: %s\n"
			  "Date: %s\n"
			  "Message-ID: %u\n"
			  "References: %u\n"
			  "\n",
			  poster, mimetype, subject, datestr, tid, replytid);
	r = write(fd, linebuf, linelen);
	CR2LF(postp, postlen);
	r = write(fd, postp, postlen);
	r = write(fd, "\n", 1);
	SYS_fsync(fd);
	close(fd);

	return 0;
}

void
rcv_news_post_thread (struct htlc_conn *htlc)
{
	u_int8_t slen, mlen, subject[256], mimetype[256], *postp = 0;
	u_int16_t postlen = 0;
	u_int32_t replytid = 0, partid = 0;
	char rlpath[MAXPATHLEN], path[MAXPATHLEN];
	int err;

	dh_start(htlc)
		switch (dh_type) {
			case HTLC_DATA_NEWS_DIR:
				if ((err = cat_to_path(dh, hxd_cfg.paths.newsdir, rlpath, path))) {
					snd_strerror(htlc, err);
					return;
				}
				break;
			case HTLC_DATA_NEWS_PARENTTHREADID:
				dh_getint(partid);
				break;
			case HTLC_DATA_NEWS_MIMETYPE:
				mlen = dh_len > 0xff ? 0xff : dh_len;
				memcpy(mimetype, dh_data, mlen);
				mimetype[mlen] = 0;
				break;
			case HTLC_DATA_NEWS_SUBJECT:
				slen = dh_len > 0xff ? 0xff : dh_len;
				memcpy(subject, dh_data, slen);
				subject[slen] = 0;
				break;
			case HTLC_DATA_NEWS_POST:
				postlen = dh_len;
				postp = dh_data;
				break;
			case HTLC_DATA_NEWS_THREADID:
				dh_getint(replytid);
				break;
		}
	dh_end()
	err = write_newsfile(rlpath, partid, replytid, htlc->name, mimetype, subject, postlen, postp);
	if (err)
		snd_errorstr(htlc, "Error writing post to file");
	else
		hlwrite(htlc, HTLS_HDR_TASK, 0, 0);
}

void
rcv_news_delete_thread (struct htlc_conn *htlc)
{
	dh_start(htlc)
	dh_end()
}

void
rcv_news_delete (struct htlc_conn *htlc)
{
	dh_start(htlc)
	dh_end()
}

void
rcv_news_mkdir (struct htlc_conn *htlc)
{
	dh_start(htlc)
	dh_end()
}

void
rcv_news_mkcategory (struct htlc_conn *htlc)
{
	dh_start(htlc)
	dh_end()
}
